### install latest version of R meteo package ###
install.packages("meteo", repos="http://R-Forge.R-project.org")

### load packages ###
library(spacetime)
library(gstat)
library(rgdal)
library(plotGoogleMaps)
library(hexbin)
library(zoo)
library(raster)
library(meteo)
library(doParallel)
library(GSODR)
library(ggplot2)
library(lubridate)
library(plyr)
library(caret)
library(RColorBrewer)
library(colorRamps)
library(ggpubr)
library(grid)

year=2008
### set working directory (that contains dem_twi and HRclim2008 directories) ###
wd="/media/sekulic/e42d0594-47d8-4b72-99f7-bf3809dbeb4f/__DOKTORAT/TAAC_Croatia/osgl/"
var="mean"
setwd(wd)

time=seq(as.Date(paste(year,"-01-01", sep="")), as.Date(paste(year,"-12-31", sep="")), by="day")
days=gsub("-","",time,fixed=TRUE)
daysNum = length(time)

border = readOGR("borders/osm/union_of_selected_boundaries_AL2-AL2.shp", layer="union_of_selected_boundaries_AL2-AL2")

################################################################################################
########### GSOD data preparing ################################################################
################################################################################################

gsod2008 <- get_GSOD(years=year) # this can take a few minutes
gsod2008 <- as.data.frame(gsod2008)
dir.create("data")
save(gsod2008, file = "data/gsod2008.rda")
load("data/gsod2008.rda")

gsod2008 = gsod2008[!is.na(gsod2008$LAT) | !is.na(gsod2008$LON), ]

### Croatian BBOX ###
lonmin=13; lonmax=20; latmin=42; latmax=47
croatia = point.in.polygon(gsod2008$LON, gsod2008$LAT, c(lonmin,lonmax,lonmax,lonmin),
                           c(latmin,latmin,latmax,latmax))
gsod2008 = gsod2008[ croatia!=0, ]

stations = gsod2008[!duplicated(gsod2008$STNID), c("STNID", "LAT", "LON", "ELEV_M")]
gsod2008 = gsod2008[, c("STNID", "TEMP", "YEARMODA")]

gsod2008stfdf <- meteo2STFDF ( obs      = gsod2008,
                               stations = stations,
                               crs      = CRS("+proj=longlat +datum=WGS84"),
                               obs.staid.time=c(1,length(gsod2008)),
                               stations.staid.lon.lat=c(1,3,2)
)

rm(gsod2008, stations, croatia, latmax, lonmax, latmin, lonmin)

### remove duplicates ###
gsod2008stfdf = rm.dupl(gsod2008stfdf, zcol = 1, zero.tol = 0)
### remove stations without any data ###
nrowsp <- length(gsod2008stfdf@sp)
numNA <- apply(matrix(gsod2008stfdf@data[,"TEMP"],
                      nrow=nrowsp,byrow=F), MARGIN=1,
               FUN=function(x) sum(is.na(x)))
rem <- numNA != daysNum
gsod2008stfdf <-  gsod2008stfdf[rem,drop=F]

### Overlay covariates ###
r <- raster("dem_twi/HR_dem.tif") # Croatian DEM
e <- extract(r,gsod2008stfdf@sp) 
gsod2008stfdf@sp$dem=e

r <- raster("dem_twi/HR_twi.tif") # Croatian TWI
e <- extract(r,gsod2008stfdf@sp)
gsod2008stfdf@sp$twi=e

rm(r, e)

data(tregcoef)
coef = as.vector(tregcoef$tmeanGSODECAD_noMODIS) # coefficients for STRK_global
### geometrical temperature trend ###
gtt <- tgeom2STFDF(gsod2008stfdf@sp, time = time, variable = var)
# identical(gtt@sp, gsod2008stfdf@sp)
### Calculate residuals for STRK_global ###
gsod2008stfdf@data$res_gsod = gsod2008stfdf@data$TEMP - (coef[1] + coef[2]*as.numeric(gtt@data$temp_geo)+coef[3]*rep(as.numeric(gsod2008stfdf@sp$dem),daysNum) + coef[4]*rep(as.numeric(gsod2008stfdf@sp$twi),daysNum))

rm(gtt)

save(gsod2008stfdf, file='data/gsod2008stfdf.rda')

################################################################################################

################################################################################################
########### Predict mean temperature for Croatia using STRK_global and GSOD ####################
################################################################################################

load("data/gsod2008stfdf.rda")

var="mean_global_gsod"
sp.nmax = 30

data(tvgms)
vario= tvgms$tmeanGSODECAD_noMODIS

i_1=c(rep(1,1),1:(daysNum -1)) ### 2 days ###
ip1=c(1:daysNum)

dir.create(paste(wd,"/",year,"/",var, sep=""))
dir.create(paste(wd,"/",year,"/",var, "/bbox", sep=""))
dir.create(paste(wd,"/",year,"/",var, "/crop", sep=""))

for (i in 1:daysNum){
  dir.create(paste(wd,"/",year,"/",var,"/",i, sep=""))
}
### list tiles (it's faster) ###
dlf = list.files(path="dem_twi/HR_tiles/dem", pattern = ".sdat")
tiles = sub(".*dem", "", sub(".sdat.*", "", dlf))
rm(dlf)
### if script stops, to check if some tiles are already processed ###
d = list.files(path=paste(year,"/",var,"/",1, sep=""), pattern = ".tif")
tiles1 = sub(".*tile", "", sub(".tif.*", "", d))
rm(d)
dif=setdiff(tiles,tiles1)
tiles=dif
rm(tiles1,dif)
### mean temperature GTT parameters ###
a <- 30.419375
b <- -15.539232

### process tiles (it could last for an hour!) ###
registerDoParallel(cores=detectCores()-1)
foreach(i = 1:length(tiles), .packages = c("raster","spacetime","gstat","rgdal","raster","doParallel","meteo")) %dopar% {
  
  dem <- readGDAL(paste('dem_twi/HR_tiles/dem/dem', tiles[i], ".sdat", sep = ""))
  r = raster(dem)
  names(dem) = 'dem'
  dem$twi<- readGDAL(paste('dem_twi/HR_tiles/twi/twi', tiles[i], ".sdat", sep = ""))$band1############################################################
  gg <- as(dem, "SpatialPixelsDataFrame")
  # gg=SpatialPointsDataFrame(coordinates(dem),data=dem@data) 
  gg@proj4string = CRS("+proj=longlat +datum=WGS84")
  rm(dem) 
  
  gtt_temp <- tgeom2STFDF(gg, time = time, variable = "mean")
  
  tlm  = coef[1] + coef[2]*as.numeric(gtt_temp@data$temp_geo)+coef[3]*rep(as.numeric(gg$dem),daysNum) + coef[4]*rep(as.numeric(gg$twi),daysNum) 
  tlm=matrix(tlm,ncol=daysNum)
  rm(gtt_temp)
  
  ################################################################################
  xxx<- lapply(1:daysNum, function(i) {
    
    obs=as(gsod2008stfdf[,i_1[i]:ip1[i],'res_gsod', drop=F],"STSDF")
    
    krigeST(as.formula("res_gsod~1"),
            data=obs,
            newdata=STF(as(gg,"SpatialPoints"),
                        gsod2008stfdf@time[i],
                        gsod2008stfdf@endTime[i]),  
            modelList=vario,
            computeVar=FALSE)$var1.pred
    
  } )
  
  res=do.call(cbind,xxx)
  rm(xxx)
  
  temp= tlm + res
  row.names(temp)<-1:nrow(gg)
  rm(tlm, res)
  
  temp=round(temp*10)
  
  for(j in 1:daysNum){
    
    pre = temp[, j]
    gg@data$pred=pre
    
    p <- try( raster( as(gg["pred"], "SpatialPixelsDataFrame")) )
    if(inherits(p, "try-error")) {
      p <- rasterize( gg, r, "pred")
    }
    writeRaster(p, paste(wd,"/",year,"/",var,"/", j, "/","tile", tiles[i], ".tif", sep = ""), "GTiff",NAflag= -32767, datatype='INT2S', overwrite=T)
    rm(pre, p)
  }
  
}
stopImplicitCluster()

### mosiac tiles ###
registerDoParallel(cores=detectCores()-1)
foreach(dd = 1:daysNum, .packages = c("raster","rgdal")) %dopar% {
  
  cat(" mosaic num: ",dd, "\n")
  TILES.sgrd.list <- dir(paste(wd,year,var,dd, sep="/"), pattern=".tif", full.names = F )
  setwd(paste(wd, year, var, dd,sep="/"))
  
  a = sub(".*tile_", "", sub(".tif.*", "", TILES.sgrd.list))
  b = sub("_.*", "", a)
  b = as.numeric(b)
  c = sub(".*_", "", a)
  c = as.numeric(c)
  a = cbind(TILES.sgrd.list,b,c)
  d = a[order(b,c),]
  TILES.sgrd.list = d[,1]
  ff =paste("\"",TILES.sgrd.list[1:length(TILES.sgrd.list)], "\"" ,sep="", collapse=" ") 
  
  system(paste("gdal_merge.py", " -ot Int16 -o ", paste(paste(wd,year, "/",var, "/", days[dd], sep=""), '.tif', sep = "" ) ," -n -32767 -a_nodata -32767 -co NUM_THREADS=ALL_CPUS -co TILED=YES -co BLOCKXSIZE=512 -co BLOCKYSIZE=512 -co COMPRESS=DEFLATE -co PREDICTOR=2 ", paste(wd, year, "/", var, "/", dd, "/", "*.tif", sep = ""), sep=""))
  
  ### crop by BBOX ###
  r = crop(raster(paste(wd,"/",year,"/",var,"/", days[dd], ".tif", sep = "")), border)
  writeRaster(r, paste(wd,"/",year,"/",var,"/bbox/", days[dd], ".tif", sep = ""), "GTiff",NAflag= -32767, datatype='INT2S', overwrite=T)
  
  r1 = mask(r, border)
  writeRaster(r1, paste(wd,"/",year,"/",var,"/crop/", days[dd], ".tif", sep = ""), "GTiff",NAflag= -32767, datatype='INT2S', overwrite=T)
  
  unlink(paste(wd,year,var,dd, sep="/"))
  unlink(paste(paste(wd,year, "/",var, "/", days[dd], sep=""), '.tif', sep = "" ))
  
}
stopImplicitCluster()

################################################################################################

################################################################################################
########### Croatian mean daily temperature data set (CMDT) preparing ##########################
################################################################################################

var="mean"
data = read.csv("HRclim2008/temp_2008.csv", header=T)
names(data) <- c('staid', 'date', 't07', 't14', 't21', 'mean')
data$t07 = ifelse(data$t07==-99.90, NA, data$t07)
data$t14 = ifelse(data$t14==-99.90, NA, data$t14)
data$t21 = ifelse(data$t21==-99.90, NA, data$t21)
data$mean = ifelse(data$mean==-99.90, NA, data$mean)
data = data[data$date != "2009-01-01", ]

sta = read.csv("HRclim2008/stations_temp_xy_2008.csv", header=T)
names(sta) = c('staid', 'name', 'code', 'lat', 'lon', 'h') 

hr_STFDF <- meteo2STFDF ( obs      = data,
                          stations = sta,
                          crs = CRS("+proj=longlat +ellps=bessel +towgs84=550.499,164.116,475.142,5.80967,2.07902,-11.62386,0.99999445824"),
                          obs.staid.time=c(1,2),
                          stations.staid.lon.lat=c(1,5,4)
)

hr_STFDF = rm.dupl(hr_STFDF, zcol = 1, zero.tol = 0)

# Remove stations out of measurements
nrowsp <- length(hr_STFDF@sp)
numNA <- apply(matrix(hr_STFDF@data[,"mean"],
                      nrow=nrowsp,byrow=F), MARGIN=1,
               FUN=function(x) sum(is.na(x)))
rem <- numNA != daysNum
hr_STFDF <-  hr_STFDF[rem,drop=F] # 1 REMOVED
### transform to WGS84 ###
hr_STFDF@sp = spTransform(hr_STFDF@sp, CRS("+proj=longlat +ellps=WGS84 +datum=WGS84"))

### OVERLAY ###
r <- raster("dem_twi/HR_dem.tif")
e <- raster::extract(r,hr_STFDF@sp)
hr_STFDF@sp$dem=e

r <- raster("dem_twi/HR_twi.tif")     
e <- raster::extract(r, hr_STFDF@sp) 
hr_STFDF@sp$twi=e

rm(r, e)
### residuals from CMDT ###
gtt <- tgeom2STFDF(hr_STFDF@sp, time = seq(as.POSIXct(paste(year,"-01-01", sep="")), as.POSIXct(paste(year,"-12-31", sep="")), 
                                           by="day"), variable = var )

data(tregcoef)
coef = as.vector(tregcoef$tmeanGSODECAD_noMODIS) # coefficients for STRK_global

hr_STFDF@data$gtt = gtt@data$temp_geo
hr_STFDF@data$gl_trend = coef[1] + coef[2]*as.numeric(gtt@data$temp_geo)+coef[3]*rep(as.numeric(hr_STFDF@sp$dem),daysNum) + coef[4]*rep(as.numeric(hr_STFDF@sp$twi),daysNum)
hr_STFDF@data$gl_res  =hr_STFDF@data$mean -  hr_STFDF@data$gl_trend

save(hr_STFDF, file=paste('data/hr_stfdf.rda', sep=""))

################################################################################################

################################################################################################
########### Identification of duplicates or stations without DEM from CMDT #####################
################################################################################################

load("data/gsod2008stfdf.rda")
load(file=paste('data/hr_stfdf.rda', sep=""))

gsod2008stfdf@sp@proj4string = hr_STFDF@sp@proj4string
border@proj4string = hr_STFDF@sp@proj4string
dupl = zerodist2(gsod2008stfdf@sp, hr_STFDF@sp, zero=2)

removed_h = hr_STFDF@sp[unique(which(is.na(hr_STFDF@sp$dem))),] # 9 stations without DEM
removed = hr_STFDF@sp[unique(dupl[,2]),]
removed = removed[!is.na(removed@data$dem), ] # 37 duplicated stations
# removed$name
# removed_h$name

### plot stations ###
dir.create("plot")

tiff("plot/Fig1.tiff", width = 84, height = 95, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig1.jpeg", width = 84, height = 95, units = 'mm', res = 1200)
ggplot() + # watch out for attribute name color order
  geom_polygon(data = border, aes(x = long, y = lat, group = group), alpha = 0.8, color = "black", fill="white", size = 0.05) +
  geom_point(data = as.data.frame(gsod2008stfdf@sp), aes(x = lon, y = lat, fill = "deepskyblue", shape = as.factor("dem")), size = 2.5, stroke = 0.2) +
  geom_point(data = as.data.frame(hr_STFDF[-unique(c(dupl[,2], which(is.na(hr_STFDF@sp$dem)))), ]@sp), aes(x = lon, y = lat, fill = "green", shape = as.factor("h")), size = 1.5, stroke = 0.2) +
  geom_point(data = as.data.frame(removed), aes(x = lon, y = lat, fill = "orange", shape = as.factor("staid")), size = 1.5, stroke = 0.2) +
  geom_point(data = as.data.frame(removed_h), aes(x = lon, y = lat, fill = "red", shape = as.factor("twi")), size = 1.5, stroke = 0.2) +
  theme(plot.title = element_text(hjust = 0.5),
        axis.text = element_text(size = 7.5),
        axis.title = element_text(size = 7.5),
        text = element_text(size = 7.5),
        legend.position = "bottom",
        legend.direction = "horizontal",
        legend.key.size= unit(0.2, "cm"),
        legend.margin = unit(0, "cm"),
        legend.title = element_text(size=7.5, face="bold")) +
  labs(x = "Longitude", y = "Latitude") +
  scale_x_continuous(limits = c(border@bbox[1,1], border@bbox[1,2])) +
  scale_y_continuous(limits = c(border@bbox[2,1], border@bbox[2,2])) +
  scale_fill_manual(name = "Stations",
                      labels = c("GSOD", "HR", "HR duplicates", "HR no DEM"),
                      values = c("deepskyblue"="deepskyblue", "green"="green", "orange"="orange", "red"="red")) +
  scale_shape_manual(name = "Stations",
                     labels = c("GSOD", "HR", "HR duplicates", "HR no DEM"),
                     values = c(21, 22, 23, 24))
dev.off()

################################################################################################

################################################################################################
########### Test STRK_global (made by GSOD stations) with CMDT #################################
################################################################################################

load("data/gsod2008stfdf.rda")
load(file=paste('data/hr_stfdf.rda', sep=""))

dupl = zerodist2(gsod2008stfdf@sp, hr_STFDF@sp, zero=2)
### remove 46 stations (duplicates or without DEM) ###
hr_STFDF_test = hr_STFDF[-unique(c(dupl[,2], which(is.na(hr_STFDF@sp$dem)))), ]

stNum = length(hr_STFDF_test@sp)
residuals = matrix(nrow = daysNum, ncol = stNum)
### extract predicted values (from GSOD) ###
for (i in 1:daysNum){
  r = raster(paste("2008/mean_global_gsod/bbox/", days[i], ".tif", sep=""))
  st_day = hr_STFDF_test[, i]
  st_day$global <- raster::extract(r, st_day) / 10
  residuals[i,] = st_day$mean - st_day$global
}

save(residuals, file = "data/res_hr_global_test2008.rda")
load("data/res_hr_global_test2008.rda")

hr_STFDF_test@data$gl_res_test = as.vector(residuals)

### total RMSE, RMSE by station, max, min, h ###

rmse_total = sqrt(sum((residuals)^2, na.rm = T)/(length(residuals[!is.na(residuals)])))
# 2.090885

data = hr_STFDF_test@data[complete.cases(hr_STFDF_test@data[, c("mean", "gl_res_test")]),]
tss = t(data$mean - mean(data$mean)) %*% (data$mean - mean(data$mean))
ess = t(data$gl_res_test) %*% (data$gl_res_test)
r2 = (tss-ess)/tss
# 0.9290643

results = matrix(nrow = stNum, ncol = 5)

for (st in 1:stNum){
  rmse = sqrt(sum((residuals[,st])^2, na.rm = T)/(length(residuals[!is.na(residuals[, st]), st])))
  max_res = max(residuals[,st], na.rm = T)
  min_res = min(residuals[,st], na.rm = T)
  h = hr_STFDF_test[st, , drop=F]@sp$h
  dem = hr_STFDF_test[st, , drop=F]@sp$dem
  results[st,] = c(rmse, max_res, min_res, h, dem)
}

colnames(results) = c("RMSE", "Maximum residual", "Minimum residual", "h on station", "dem")

# save(results, file="data/results_global_test2008.rda")
load("data/results_global_test2008.rda")

results_order = results[order(results[,1]),]

head(results_order, 20)
tail(results_order, 30)

hr_STFDF_test@sp$rmse = results[,1]
hr_STFDF_test@sp$max_res = results[,2]
hr_STFDF_test@sp$min_res = results[,3]
hr_STFDF_test@sp$h = results[,4]
hr_STFDF_test@sp$dem = results[,5]

global_plot = hr_STFDF_test@sp[!is.na(hr_STFDF_test@sp$rmse), c(1, 2, 4, 5, 7, 8, 9)]
global_plot$rmse = round(global_plot$rmse, 1)
global_plot = global_plot[order(global_plot@data$rmse, decreasing = T), ]
bubbles_global = bubbleSP(global_plot, zcol="rmse", scale_e = 100)
dir.create("html")
setwd("html/")
m = plotGoogleMaps(bubbles_global, filename = "global_test2008.html", zcol = "rmse")
setwd(wd)

dem <- raster("dem_twi/HR_dem.tif")
dem <- mask(dem, border)
twi <- raster("dem_twi/HR_twi.tif")
twi <- mask(twi, border)

### plot DEM and TWI ###
tiff("plot/Fig2.tiff", width = 174, height = 75, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig2.jpeg", width = 174, height = 75, units = 'mm', res = 1200)
par(mfrow=c(1,2), cex = 0.7, mar=c(4.5,4.5,1,0.5))
plot(dem, col = terrain.colors(255), xlab = "Longitude", ylab = "Latitude") #, cex.axis = 0.5, cex.lab = 0.5)
plot(border, lwd=0.3, add=T)
par(cex = 0.7)
plot(twi, col = rev(terrain.colors(255)), xlab = "Longitude", cex = 0.5) #, cex.axis = 0.5, cex.lab = 0.5)
plot(border, lwd=0.3, add=T)
dev.off()

par(mfrow=c(1,1))

### plot test RMSEs per station ###
tiff("plot/Fig5.tiff", width = 84, height = 84, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig5.jpeg", width = 84, height = 84, units = 'mm', res = 1200)
ggplot(as.data.frame(global_plot), aes(x = lon, y = lat, size = rmse)) +
  geom_polygon(data = border, aes(x = long, y = lat, group = group), alpha = 0.8, color = "black", fill="white", size = 0.1) +
  geom_point(shape = 21, colour = "mediumvioletred", fill = "springgreen",alpha=.8) +
  theme(plot.title = element_text(hjust = 0.5),
        axis.text = element_text(size = 7, color="black"),
        axis.title = element_text(size = 7),
        text = element_text(size = 7),
        legend.position = c(.98, .35),
        legend.justification = c("right", "bottom"),
        legend.direction = "horizontal",
        legend.key.size= unit(0.2, "cm"),
        legend.margin = unit(0, "cm"),
        legend.title = element_text(size=7, face="bold")) +
  labs(x = "Longitude", y = "Latitude") + labs(size = "RMSE") +
  scale_size_identity(guide="legend", breaks = c(1, 2, 3, 4, 5, 6))
dev.off()

################################################################################################

################################################################################################
########### LOO cross-validation for STRK_global with CMDT #####################################
################################################################################################

load(file=paste('data/hr_stfdf.rda', sep=""))

sp.nmax = 30
i_1=c(1,1:(daysNum -1))
ip1=1:daysNum

temp = hr_STFDF

nrowsp <- length(temp@sp)
# count NAs per stations
numNA <- apply(matrix(temp@data[,'gl_res'],
                      nrow=nrowsp,byrow=F), MARGIN=1,
               FUN=function(x) sum(is.na(x)))
# Remove stations out of covariates
rem <- numNA != length(time)
temp <-  temp[rem,drop=F]
N_POINTS <- length(temp@sp@coords[,1])

registerDoParallel(cores=detectCores()-1)
cv <- foreach(i = 1:N_POINTS, .packages = c("raster","spacetime","gstat","rgdal","meteo")) %dopar% {
  
  st= temp@sp
  st$dist=spDists(temp@sp,temp@sp[i,])
  tmp_st<-st[ order(st$'dist') , ]
  ### remove target station. Instead of sp.nmax all of the stations can be used, but accuracy stays the same ###
  local_t= row.names(tmp_st[2:sp.nmax,] )
  
  xxx = as.list ( rep(NA, length(time) ) )
  for( ii in 1:length(time) ) {
    data=temp[local_t, i_1[ii]:ip1[ii],'gl_res',drop=F]
    nrowsp <- length(data@sp)
    # count NAs per stations
    numNA <- apply(matrix(data@data[,'gl_res'],
                          nrow=nrowsp,byrow=F), MARGIN=1,
                   FUN=function(x) sum(is.na(x)))
    # Remove stations out of covariates
    rem <- !numNA > 0
    data <-  data[rem,drop=F]
    
    xxx[[ii]]=krigeST(as.formula("gl_res~1"),
                      data=data, 
                      newdata=STF(as(temp@sp[i,],"SpatialPoints"),
                                  temp@time[ii],  
                                  temp@endTime[ii]),     
                      modelList=vario,
                      computeVar=FALSE)@data[,1]
  } # end of  for
  ret=unlist(xxx) 
}
stopImplicitCluster()

cv <- do.call(rbind,cv)
cv <- as.vector(cv)
cv.temp_gl <- temp
cv.temp_gl$pred.cv <- cv + cv.temp_gl$gl_trend
cv.temp_gl$resid.cv <- cv.temp_gl$mean  - cv.temp_gl@data$pred.cv

save(cv.temp_gl, file=paste('data/cv_global_loo.rda', sep=""))
load(paste('data/cv_global_loo.rda', sep=""))

rmse_total = sqrt(sum((cv.temp_gl$resid.cv)^2, na.rm = T)/(length(cv.temp_gl$resid.cv[!is.na(cv.temp_gl$resid.cv)])))
# 1.85451

data = cv.temp_gl@data[complete.cases(cv.temp_gl@data[, c(4, 9)]),]
tss = t(data$mean - mean(data$mean)) %*% (data$mean - mean(data$mean))
ess = t(data$resid.cv) %*% (data$resid.cv)
r2 = (tss-ess)/tss
# 0.9439575

### per month ###
for (m in 1:12){
  all <- cv.temp_gl[, month(cv.temp_gl@time)==m ]
  rmse = sqrt(sum((all$resid.cv)^2, na.rm = T)/(length(all$resid.cv[!is.na(all$resid.cv)])))
  max = max(all$mean, na.rm = T)
  min = min(all$mean, na.rm = T)
  range = max - min
  print(round(c(rmse, max, min, range),2))
}

stNum = length(cv.temp_gl@sp)
results = matrix(nrow = stNum, ncol = 5)

for (st in 1:stNum){
  rmse = sqrt(sum((cv.temp_gl[st, ]$resid.cv)^2, na.rm = T)/(length(cv.temp_gl[st, ]$resid.cv[!is.na(cv.temp_gl[st, ]$resid.cv)])))
  max = max(cv.temp_gl[st, ]$resid.cv, na.rm = T)
  min = min(cv.temp_gl[st, ]$resid.cv, na.rm = T)
  h = cv.temp_gl[st, , drop=F]@sp$h
  dem = cv.temp_gl[st, , drop=F]@sp$dem
  results[st,] = c(rmse, max, min, h, dem)
}

colnames(results) = c("rmse", "max", "min", "h", "dem")

# save(results, file=paste('data/res_cv_global_loo.rda', sep = ""))
load(paste('data/res_cv_global_loo.rda', sep = ""))

results_order = results[order(results[,1]),]

head(results_order, 20)
tail(results_order, 20)

cv.temp_gl@sp$rmse = results[,1]
cv.temp_gl@sp$max = results[,2]
cv.temp_gl@sp$min = results[,3]

global_loo_plot = cv.temp_gl@sp[!is.na(cv.temp_gl@sp$rmse),]
global_loo_plot$rmse = round(global_loo_plot$rmse, 1)
global_loo_plot = global_loo_plot[order(global_loo_plot@data$rmse, decreasing = T), ]
bubbles_global_loo = bubbleSP(global_loo_plot, zcol="rmse", scale_e = 100)
setwd("html/")
m = plotGoogleMaps(bubbles_global_loo, filename = paste('cv_global_loo.html', sep = ""), zcol = "rmse")
setwd(wd)

save(cv.temp_gl, file=paste('data/cv_global_loo.rda', sep=""))
load(paste('data/cv_global_loo.rda', sep=""))

################################################################################################

################################################################################################
########### 5-fold stratified cross-validation for STRK_global with CMDT #######################
################################################################################################

load(file=paste('data/hr_stfdf.rda', sep=""))

source("stratfolds.R")
test = hr_STFDF@data
test$ID = rep(hr_STFDF@sp$staid, daysNum)
test$weight <-  rep(1,dim(hr_STFDF@data)[1])
test$dem = rep(as.numeric(hr_STFDF@sp$dem),daysNum)
test$lon = rep(as.numeric(hr_STFDF@sp@coords[, "lon"]),daysNum)
test$lat = rep(as.numeric(hr_STFDF@sp@coords[, "lat"]),daysNum)
strat <- stratfold3d(target.name = "dem",
                     other.names = c("lon","lat"),
                     data = test[1:157,],
                     num.folds = 5,
                     num.means = 5,
                     seed = 41,
                     cum.prop = 0.9)

save(strat, file="data/folds.rda")
load(file="data/folds.rda")

# ### f-ja za plot ###
# plotfolds<-function(folds,targetvar){
#   allData<-folds$data
#   targetVar<-targetvar
#   allData.unique<-ddply(allData,.(ID),here(summarize),target=mean(eval(parse(text=targetVar))),longitude=lon[1],latitude=lat[1],fold=fold[1])
#   q <- ggplot(allData.unique,aes(x = longitude, y = latitude))
#   r <- q +geom_point(aes(size = target), pch = 21, alpha=0.5) + scale_size_continuous(range=c(1,10))
#   r <- r + facet_wrap(~ fold)
#   r <- r + aes(fill = fold) + labs(size = "DEM") + labs(fill = "Fold")
#   plot(r)
# }
# 
# # tiff("plot/stratification.tiff", width = 20, height = 20, units = 'cm', res = 300)
# # jpeg("plot/stratification.jpeg", width = 20, height = 20, units = 'cm', res = 300)
# plotfolds(folds = strat, targetvar = "dem")
# # dev.off()

data <- strat$data
by(data$mean, data$fold, length)
by(data$dem, data$fold, summary)

### boxplot ###
tiff("plot/Fig8.tiff", width = 84, height = 84, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig8.jpeg", width = 84, height = 84, units = 'mm', res = 1200)
ggplot(data = data, aes(x=fold, y=dem, fill=fold)) +
  geom_boxplot(show.legend = F) +
  theme(plot.title = element_text(hjust = 0.5),
        axis.text = element_text(size = 7, color="black"),
        axis.title = element_text(size = 7),
        text = element_text(size = 7)) +
  xlab("Fold") +
  ylab("DEM [m]")
dev.off()

sp.nmax = 30
i_1=c(1,1:(daysNum -1))
ip1=1:daysNum

for(val_fold in 1:5){
  val_ids = as.vector(unlist(strat$obs.fold.list[val_fold]))
  dev_ids = as.vector(unlist(strat$obs.fold.list[-val_fold]))
  
  hr_STFDF@sp$ind = ifelse(index(hr_STFDF@sp) %in% dev_ids, 1, ifelse(index(hr_STFDF@sp) %in% val_ids, 2, NA))
  dev = hr_STFDF[hr_STFDF@sp$ind==1, ]
  val = hr_STFDF[hr_STFDF@sp$ind==2, ]
  
  temp = val
  
  nrowsp <- length(temp@sp)
  # count NAs per stations
  numNA <- apply(matrix(temp@data[,'gl_res'],
                        nrow=nrowsp,byrow=F), MARGIN=1,
                 FUN=function(x) sum(is.na(x)))
  # Remove stations out of covariates
  rem <- numNA != length(time)
  temp <-  temp[rem,drop=F]
  
  N_POINTS <- length(temp@sp@coords[,1])
  
  registerDoParallel(cores=detectCores()-1)
  cv <- foreach(i = 1:N_POINTS, .packages = c("raster","spacetime","gstat","rgdal","meteo")) %dopar% {
    st= dev@sp
    st$dist=spDists(dev@sp,temp@sp[i,])
    tmp_st<-st[ order(st$'dist') , ]
    # now the 1st station is added because of independent CV
    local_t= row.names(tmp_st[1:sp.nmax,] ) 
    
    xxx = as.list ( rep(NA, length(time) ) )
    for( ii in 1:length(time) ) {
      data=dev[local_t, i_1[ii]:ip1[ii],'gl_res',drop=F]
      nrowsp <- length(data@sp)
      # count NAs per stations
      numNA <- apply(matrix(data@data[,'gl_res'],
                            nrow=nrowsp,byrow=F), MARGIN=1,
                     FUN=function(x) sum(is.na(x)))
      # Remove stations out of covariates
      rem <- !numNA > 0
      data <-  data[rem,drop=F]
      
      xxx[[ii]]=krigeST(as.formula("gl_res~1"),
                        data=data, 
                        newdata=STF(as(temp@sp[i,],"SpatialPoints"),
                                    temp@time[ii],  
                                    temp@endTime[ii]),     
                        modelList=vario,
                        computeVar=FALSE)@data[,1]
    } # end of  for
    
    ret=unlist(xxx) 
  }
  stopImplicitCluster()
  
  cv <- do.call(rbind,cv)
  cv <- as.vector(cv)
  cv.temp <- temp
  cv.temp$pred.cv <- cv + cv.temp$gl_trend
  cv.temp$resid.cv <- cv.temp$mean  - cv.temp@data$pred.cv
  
  save(cv.temp, file=paste('data/cv_global_fold_', val_fold, '.rda', sep=""))
  load(paste('data/cv_global_fold_', val_fold, '.rda', sep=""))
  
  rmse_total = sqrt(sum((cv.temp$resid.cv)^2, na.rm = T)/(length(cv.temp$resid.cv[!is.na(cv.temp$resid.cv)])))
  print(rmse_total)
  # 1.701397 # 1.872712 # 2.238707 # 1.899263 # 1.597324

  data = cv.temp@data[complete.cases(cv.temp@data[, c(4, 7)]),]
  tss = t(data$mean - mean(data$mean)) %*% (data$mean - mean(data$mean))
  ess = t(data$resid.cv) %*% (data$resid.cv)
  r2 = (tss-ess)/tss
  print(r2)
  # 0.9529108 # 0.9430985 # 0.920598 # 0.9405573 # 0.9574322
  
  stNum = length(cv.temp@sp)
  results = matrix(nrow = stNum, ncol = 5)
  
  for (st in 1:stNum){
    rmse = sqrt(sum((cv.temp[st, ]$resid.cv)^2, na.rm = T)/(length(cv.temp[st, ]$resid.cv[!is.na(cv.temp[st, ]$resid.cv)])))
    max = max(cv.temp[st, ]$resid.cv, na.rm = T)
    min = min(cv.temp[st, ]$resid.cv, na.rm = T)
    h = cv.temp[st, , drop=F]@sp$h
    dem = cv.temp[st, , drop=F]@sp$dem
    results[st,] = c(rmse, max, min, h, dem)
  }
  colnames(results) = c("rmse", "max", "min", "h", "dem")
  
  save(results, file=paste('data/cv_global_per_station_', val_fold, '.rda', sep = ""))
  load(paste('data/cv_global_per_station_', val_fold, '.rda', sep = ""))
  
  results_order = results[order(results[,1]),]
  print(results_order)
  
  cv.temp@sp$rmse = results[,1]
  cv.temp@sp$max = results[,2]
  cv.temp@sp$min = results[,3]
  cv.temp@sp$h = results[,4]
  cv.temp@sp$dem = results[,5]
  
}

# per month #
for (m in 1:12){
  all_res <- c()
  for (i in 1:5) {
    val_fold = i
    load(paste('data/cv_global_fold_', val_fold, '.rda', sep=""))
    all_res <- c(all_res, cv.temp[, month(cv.temp@time)==m ]$resid.cv)
  }
  rmse = sqrt(sum((all_res)^2, na.rm = T)/(length(all_res[!is.na(all_res)])))
  print(round(rmse,2))
}

################################################################################################

################################################################################################
########### Creation of STRK_Croatia using CMDT ################################################
################################################################################################

load(file=paste('data/hr_stfdf.rda', sep=""))

temp_df = hr_STFDF[, ]@data$mean
temp_df = data.frame(temp_df)
names(temp_df) <- "mean"
temp_df$gtt = hr_STFDF@data$gtt
temp_df$dem = rep(as.numeric(hr_STFDF@sp$dem),daysNum)
temp_df$twi = rep(as.numeric(hr_STFDF@sp$twi),daysNum)
temp_df$completed = complete.cases(temp_df)

summary(temp_df)
summary(complete.cases(temp_df))

### multiple linear regression - trend ###
temp_df.dev = temp_df[complete.cases(temp_df), ]

set.seed(42)
lm = lm(mean ~ gtt + dem + twi, temp_df.dev)
coefficients(lm) # model coefficients, added to tregcoef$tmeanHR
summary(lm)
confint(lm, level=0.95) # CIs for model parameters 
anova(lm) # anova table 
vcov(lm) # covariance matrix for model parameters

save(lm, file=paste('data/hr_lm', '.rda', sep=""))
load(file=paste('data/hr_lm', '.rda', sep=""))

rmse = sqrt(sum((temp_df.dev$mean - lm$fitted.values)^2)/(length(temp_df.dev$mean)))
# 3.491122
tss = t(temp_df.dev$mean - mean(temp_df.dev$mean)) %*% (temp_df.dev$mean - mean(temp_df.dev$mean))
ess = t(temp_df.dev$mean - lm$fitted.values) %*% (temp_df.dev$mean - lm$fitted.values)
r2 = (tss-ess)/tss
# 0.8013956

hr_trend = c()
br = 1
for (i in 1:length(temp_df$completed)) {
  if (temp_df$completed[i]){
    hr_trend[i] = lm$fitted.values[br]
    br = br + 1
  }
}
hr_STFDF$hr_trend = hr_trend

### fitting spatio-temporal variogram ###

hr_STFDF$hr_res = hr_STFDF$mean - hr_STFDF$hr_trend

save(hr_STFDF, file = "data/hr_stfdf.rda")
load(file = "data/hr_stfdf.rda")

tiff("plot/Fig3a.tiff", width = 80, height = 75, units = 'mm', res = 1200, compression = "lzw")
jpeg("plot/Fig3a.jpeg", width = 80, height = 75, units = 'mm', res = 1200)
p1 <- hexbinplot(hr_STFDF@data$mean~hr_STFDF@data$hr_trend, xlab="", ylab="",
           type="r", col.line = "black", lty=2, lwd=1, xlim=c(-20,40), ylim=c(-20,40), cex.labels = 0.7, cex.title = 0.7)
p2 <- plot(p1, xlab="", ylab="", lcex=.7)
grid.text("Linear model [°C]", .48, .06, gp=gpar(fontsize=8.7))
grid.text("Mean daily temperature [°C]", .05, .55, rot=90, gp=gpar(fontsize=8.7))
dev.off()

tiff("plot/Fig3b.tiff", width = 80, height = 75, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig3b.jpeg", width = 80, height = 75, units = 'mm', res = 1200)
par(cex = 0.7)
res_hist = hist(hr_STFDF$hr_res, main=NULL, xlab="Residuals [°C]", breaks=50)
multiplier <- res_hist$counts / res_hist$density
hist_density <- density(hr_STFDF$hr_res, na.rm=T)
hist_density$y <- hist_density$y * multiplier[1]
lines(hist_density, col="red", lwd=2) 
dev.off()

var = variogramST(hr_res ~ 1, hr_STFDF, tlags = 0:5, cutoff = 300, width = 10, na.omit=T) # tunit="days"
var$dist = var$dist/100
var$spacelag = var$spacelag/100
var$avgDist = var$avgDist/100
attr(var, "boundaries") = attr(var, "boundaries") / 100
var$timelag = var$timelag/24
attr(var$timelag, "units") = "days"
# plot(var, map = F)#, main="2d sample variogram")

estiStAni(var, c(0, 10), "metric",
          vgm(psill=4,"Sph", range=3, nugget=0),
          vgm(psill=8,"Sph", range=3, nugget=0) )

pars.l <- c(sill.s = 0, range.s = 1, nugget.s = 0,
            sill.t = 0, range.t = 1, nugget.t = 0,
            sill.st = 0, range.st = 1, nugget.st = 0,
            anis = 0)

sumMetric <- vgmST("sumMetric",
                   space = vgm(psill=3,"Sph", range=3, nugget=0.1),
                   time = vgm(psill=7,"Sph", range=3, nugget=0.1),
                   joint = vgm(psill=1,"Sph", range=2, nugget=0.1),
                   stAni=2)

hr_sumMetric_vgm <- fit.StVariogram(var, sumMetric, method="L-BFGS-B",lower=pars.l)
attr(hr_sumMetric_vgm, "MSE")
hr_sumMetric_vgm

var$dist <- var$dist*100
var$spacelag <- var$spacelag*100
var$avgDist = var$avgDist*100
attr(var, "boundaries") = attr(var, "boundaries") * 100

hr_sumMetric_vgm$space$range <- hr_sumMetric_vgm$space$range*100
hr_sumMetric_vgm$joint$range <- hr_sumMetric_vgm$joint$range*100
hr_sumMetric_vgm$stAni <- hr_sumMetric_vgm$stAni*100

# plot(var, hr_sumMetric_vgm,map=F)#, main="2d fitted sum-metric variogram")

tiff("plot/Fig4a.tiff", width = 84, height = 75, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig4a.jpeg", width = 84, height = 75, units = 'mm', res = 1200)
plot(var, wireframe=T, zlim=c(0,12),
     zlab=NULL,
     xlab=list("distance (km)", rot=30, cex=0.7),
     ylab=list("time lag (days)", rot=-35, cex=0.7),
     scales=list(arrows=F, z = list(distance = 5), cex=0.5)) #, main="3d sample variogram")
dev.off()

tiff("plot/Fig4b.tiff", width = 84, height = 75, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig4b.jpeg", width = 84, height = 75, units = 'mm', res = 1200)
plot(var, list(hr_sumMetric_vgm),all=T, wireframe=T, zlim=c(0,12),
     zlab=NULL,
     xlab=list("distance (km)", rot=30, cex=0.7),
     ylab=list("time lag (days)", rot=-35, cex=0.7),
     scales=list(arrows=F, z = list(distance = 5), cex=0.5))#, main="3d fitted sum-metric variogram")
dev.off()

save(hr_sumMetric_vgm, file=paste('data/hr_sumMetric_vgm', '.rda', sep=""))
load(file=paste('data/hr_sumMetric_vgm', '.rda', sep=""))

################################################################################################

################################################################################################
########### LOO cross-validation for STRK_Croatia with CMDT ####################################
################################################################################################

load(file=paste('data/hr_stfdf.rda', sep=""))

sp.nmax = 30
i_1=c(rep(1,6),1:(daysNum -6))
ip1=c(1:daysNum) ### 7 days (temporal range) ###

data(tvgms)
# identical(tvgms$tmeanHR, hr_sumMetric_vgm)
vario = tvgms$tmeanHR # hr_sumMetric_vgm
temp = hr_STFDF

nrowsp <- length(temp@sp)
# count NAs per stations
numNA <- apply(matrix(temp@data[,'hr_res'],
                      nrow=nrowsp,byrow=F), MARGIN=1,
               FUN=function(x) sum(is.na(x)))
# Remove stations out of covariates
rem <- numNA != length(time)
temp <-  temp[rem,drop=F]

N_POINTS <- length(temp@sp@coords[,1])

registerDoParallel(cores=detectCores()-1)
cv <- foreach(i = 1:N_POINTS, .packages = c("raster","spacetime","gstat","rgdal","raster","doParallel","snowfall","meteo")) %dopar% {
  
  st= temp@sp
  st$dist=spDists(temp@sp,temp@sp[i,])
  tmp_st<-st[ order(st$'dist') , ]
  local_t= row.names(tmp_st[2:sp.nmax,] ) # remove target station
  
  xxx = as.list ( rep(NA, length(time) ) )
  for( ii in 1:length(time) ) {
    data=temp[local_t, i_1[ii]:ip1[ii],'hr_res',drop=F]
    nrowsp <- length(data@sp)
    # count NAs per stations
    numNA <- apply(matrix(data@data[,'hr_res'],
                          nrow=nrowsp,byrow=F), MARGIN=1,
                   FUN=function(x) sum(is.na(x)))
    # Remove stations out of covariates
    rem <- !numNA > 0
    data <-  data[rem,drop=F]
    
    xxx[[ii]]=krigeST(as.formula("hr_res~1"),
                      data=data, 
                      newdata=STF(as(temp@sp[i,],"SpatialPoints"),
                                  temp@time[ii],  
                                  temp@endTime[ii]),     
                      modelList=vario,
                      computeVar=FALSE)@data[,1]
  } # end of  for
  ret=unlist(xxx) 
}
stopImplicitCluster()

cv <- do.call(rbind,cv)
cv <- as.vector(cv)
cv.temp_hr <- temp

cv.temp_hr$pred.cv <- cv + cv.temp_hr$hr_trend
cv.temp_hr$resid.cv <- cv.temp_hr$mean  - cv.temp_hr@data$pred.cv

save(cv.temp_hr, file=paste('data/cv_hr_loo.rda', sep=""))
load(paste('data/cv_hr_loo.rda', sep=""))

rmse_total = sqrt(sum((cv.temp_hr$resid.cv)^2, na.rm = T)/(length(cv.temp_hr$resid.cv[!is.na(cv.temp_hr$resid.cv)])))
# 1.151294

data = cv.temp_hr@data[complete.cases(cv.temp_hr@data[, c(4, 7)]),]
tss = t(data$mean - mean(data$mean)) %*% (data$mean - mean(data$mean))
ess = t(data$resid.cv) %*% (data$resid.cv)
r2 = (tss-ess)/tss
# 0.9784011

# per month #
for (m in 1:12){
  rmse = sqrt(sum((cv.temp_hr[, month(cv.temp_hr@time)==m ]$resid.cv)^2, na.rm = T)/(length(cv.temp_hr[, month(cv.temp_hr@time)==m ]$resid.cv[!is.na(cv.temp_hr[, month(cv.temp_hr@time)==m ]$resid.cv)])))
  print(round(rmse,2))
}

### plot predictions and observations at 2 stations ###
load(paste('data/cv_global_loo.rda', sep=""))
a <- cv.temp_gl[cv.temp_gl@sp$dem==1514,]$pred.cv
n1 <- as.character(cv.temp_gl@sp[cv.temp_gl@sp$dem==1514, ]@data$name)
n1 <- "Zavižan"
a1 <- cv.temp_gl[cv.temp_gl@sp$dem==121,]$pred.cv
n2 <- as.character(cv.temp_gl@sp[cv.temp_gl@sp$dem==121, ]@data$name)
load(paste('data/cv_hr_loo.rda', sep=""))
b <- cv.temp_hr[cv.temp_hr@sp$dem==1514,]$pred.cv
b1 <- cv.temp_hr[cv.temp_hr@sp$dem==121,]$pred.cv
c <- cv.temp_hr[cv.temp_hr@sp$dem==1514,]$mean
c1 <- cv.temp_hr[cv.temp_hr@sp$dem==121,]$mean

summary(a-b)
summary(a1-b1)

Sys.setlocale("LC_TIME", "C")
tiff("plot/Fig10.tiff", width = 174, height = 90, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig10.jpeg", width = 174, height = 90, units = 'mm', res = 1200)
par(mfrow=c(2,1), cex = 0.7, mar=c(4,4,1,1))
plot.zoo(cbind(c1, a1, b1), 
         plot.type = "single", 
         col = c("chartreuse3", "red", "blue"),
         ylab = "Residual [°C]", xlab = "", main = paste(n2, " (H = 121 m)"),
         xlim = NULL)#c("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"))
abline(h=0, lty=2)
legend("topright", inset=c(0,0), y.intersp = 1, legend = c("Observation", "STRK_global", "STRK_Croatia"),  lty = 1, bty = "n", col = c("chartreuse3", "red", "blue"), cex = 1)
par(cex = 0.7)
plot.zoo(cbind(c, a, b), 
         plot.type = "single", 
         col = c("chartreuse3", "red", "blue"),
         ylab = "Residual [°C]", xlab = "Month", main = paste(n1, " (H = 1514 m)"),
         xlim = NULL)#c("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"))
abline(h=0, lty=2)
# legend("topright", inset=c(0,0), y.intersp = 1, legend = c("Observation", "STRK_global", "STRK_Croatia"),  lty = 1, bty = "n", col = c("chartreuse3", "red", "blue"), cex = 1)
dev.off()
par(mfrow=c(1,1))

stNum = length(cv.temp_hr@sp)
results = matrix(nrow = stNum, ncol = 5)

for (st in 1:stNum){
  rmse = sqrt(sum((cv.temp_hr[st, ]$resid.cv)^2, na.rm = T)/(length(cv.temp_hr[st, ]$resid.cv[!is.na(cv.temp_hr[st, ]$resid.cv)])))
  max = max(cv.temp_hr[st, ]$resid.cv, na.rm = T)
  min = min(cv.temp_hr[st, ]$resid.cv, na.rm = T)
  h = cv.temp_hr[st, , drop=F]@sp$h
  dem = cv.temp_hr[st, , drop=F]@sp$dem
  results[st,] = c(rmse, max, min, h, dem)
}
colnames(results) = c("rmse", "max", "min", "h", "dem")

save(results, file=paste('data/res_cv_hr_loo.rda', sep = ""))
load(paste('data/res_cv_hr_loo.rda', sep = ""))

results_order = results[order(results[,1]),]

head(results_order, 20)
tail(results_order, 20)

cv.temp_hr@sp$rmse = results[,1]
cv.temp_hr@sp$max = results[,2]
cv.temp_hr@sp$min = results[,3]

local_loo_plot = cv.temp_hr@sp[!is.na(cv.temp_hr@sp$rmse),]
local_loo_plot$rmse = round(local_loo_plot$rmse, 1)
local_loo_plot = local_loo_plot[order(local_loo_plot@data$rmse, decreasing = T), ]
bubbles = bubbleSP(local_loo_plot, zcol="rmse", scale_e = 100)
setwd("html/")
m = plotGoogleMaps(bubbles, filename = paste('cv_hr_loo.html', sep = ""), zcol = "rmse")
setwd(wd)

save(cv.temp_hr, file=paste('data/cv_hr_loo.rda', sep=""))
load(paste('data/cv_hr_loo.rda', sep=""))

tiff("plot/Fig7.tiff", width = 129, height = 65, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig7.jpeg", width = 129, height = 65, units = 'mm', res = 1200)
par(mfrow=c(1,2), cex = 0.7, mar=c(4.5,4.5,1,0.5))
plot(cv.temp_gl@sp$rmse ~ cv.temp_gl@sp$dem, xlab = "DEM", ylab = "RMSE", pch=20, cex = 0.5)
s_gl <- cv.temp_gl[cv.temp_gl@sp$dem>1000]
text(s_gl@sp$dem, s_gl@sp$rmse, labels = s_gl@sp$dem, pos = 2, col="red")
par(cex = 0.7)
plot(cv.temp_hr@sp$rmse ~ cv.temp_hr@sp$dem, xlab = "DEM", ylab = "RMSE", pch=20, cex = 0.5)
s <- cv.temp_hr[cv.temp_hr@sp$dem>1000]
text(s@sp$dem, s@sp$rmse, labels = s@sp$dem, pos = 2, col="red")
dev.off()
par(mfrow=c(1,1))

### both STRK_global and STRK_Croatia LOO RMSE per station plots ###
global_loo_plot = cv.temp_gl@sp[!is.na(cv.temp_gl@sp$rmse),]
global_loo_plot$rmse = round(global_loo_plot$rmse, 1)
global_loo_plot = global_loo_plot[order(global_loo_plot@data$rmse, decreasing = T), ]
gl_p_loo <- ggplot(as.data.frame(global_loo_plot), aes(x = lon, y = lat, size = rmse)) +
  geom_polygon(data = border, aes(x = long, y = lat, group = group), alpha = 0.8, color = "black", fill="white", size = 0.1) +
  geom_point(shape = 21, colour = "mediumvioletred", fill = "springgreen", alpha = 0.8) +
  theme(plot.title = element_text(hjust = 0.5),
        axis.text = element_text(size = 5),
        axis.title = element_text(size = 7),
        text = element_text(size = 7),
        legend.position = "bottom",
        legend.direction = "horizontal",
        legend.key.size= unit(0.2, "cm"),
        legend.margin = unit(0, "cm"),
        legend.title = element_text(size=7, face="bold")) +
  labs(x = "Longitude", y = "Latitude") + labs(size = "RMSE") +
  scale_size_identity(guide="legend", breaks = c(1, 2, 3, 4, 5, 6, 7, 8, 9))

loc_p_loo <- ggplot(as.data.frame(local_loo_plot), aes(x = lon, y = lat, size = rmse)) +
  geom_polygon(data = border, aes(x = long, y = lat, group = group), alpha = 0.8, color = "black", fill="white", size = 0.1) +
  geom_point(shape = 21, colour = "mediumvioletred", fill = "springgreen", alpha = 0.8) +
  theme(plot.title = element_text(hjust = 0.5),
        axis.text = element_text(size = 5),
        axis.title = element_text(size = 7),
        text = element_text(size = 7),
        legend.position = "bottom",
        legend.direction = "horizontal",
        legend.key.size= unit(0.2, "cm"),
        legend.margin = unit(0, "cm"),
        legend.title = element_text(size=7, face="bold")) +
  labs(x = "Longitude", y = "") + labs(size = "RMSE") +
  scale_size_identity(guide="legend", breaks = c(1, 2, 3))

tiff("plot/Fig6.tiff", width = 174, height = 105, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig6.jpeg", width = 174, height = 105, units = 'mm', res = 1200)
ggarrange(gl_p_loo, loc_p_loo, ncol=2, nrow=1, common.legend = TRUE, legend="bottom")
dev.off()

################################################################################################

################################################################################################
########### 5-fold stratified cross-validation for STRK_Croatia with CMDT ######################
################################################################################################

load(file=paste('data/hr_stfdf.rda', sep=""))

for(val_fold in 1:5){
  
  val_ids = as.vector(unlist(strat$obs.fold.list[val_fold]))
  dev_ids = as.vector(unlist(strat$obs.fold.list[-val_fold]))
  
  hr_STFDF@sp$ind = ifelse(index(hr_STFDF@sp) %in% dev_ids, 1, ifelse(index(hr_STFDF@sp) %in% val_ids, 2, NA))
  dev = hr_STFDF[hr_STFDF@sp$ind==1, ]
  val = hr_STFDF[hr_STFDF@sp$ind==2, ]
  
  sp.nmax = 30
  i_1=c(rep(1,6),1:(daysNum -6))
  ip1=c(1:daysNum)
  
  vario = tvgms$tmeanHR
  
  temp = val
  
  nrowsp <- length(temp@sp)
  # count NAs per stations
  numNA <- apply(matrix(temp@data[,'hr_res'],
                        nrow=nrowsp,byrow=F), MARGIN=1,
                 FUN=function(x) sum(is.na(x)))
  # Remove stations out of covariates
  rem <- numNA != length(time)
  temp <-  temp[rem,drop=F]
  
  N_POINTS <- length(temp@sp@coords[,1])
  
  registerDoParallel(cores=detectCores()-1)
  cv <- foreach(i = 1:N_POINTS, .packages = c("raster","spacetime","gstat","rgdal","raster","doParallel","snowfall","meteo")) %dopar% {
    st= dev@sp
    st$dist=spDists(dev@sp,temp@sp[i,])
    tmp_st<-st[ order(st$'dist') , ]
    # do not remove target station (now is added because is endependent CV)
    local_t= row.names(tmp_st[1:sp.nmax,] )
    
    xxx = as.list ( rep(NA, length(time) ) )
    for( ii in 1:length(time) ) {
      data=dev[local_t, i_1[ii]:ip1[ii],'hr_res',drop=F]
      nrowsp <- length(data@sp)
      # count NAs per stations
      numNA <- apply(matrix(data@data[,'hr_res'],
                            nrow=nrowsp,byrow=F), MARGIN=1,
                     FUN=function(x) sum(is.na(x)))
      # Remove stations out of covariates
      rem <- !numNA > 0
      data <-  data[rem,drop=F]
      
      xxx[[ii]]=krigeST(as.formula("hr_res~1"),
                        data=data, 
                        newdata=STF(as(temp@sp[i,],"SpatialPoints"),
                                    temp@time[ii],  
                                    temp@endTime[ii]),     
                        modelList=vario,
                        computeVar=FALSE)@data[,1]
    } # end of  for
    ret=unlist(xxx) 
  }
  stopImplicitCluster()
  
  cv <- do.call(rbind,cv)
  cv <- as.vector(cv)
  cv.temp <- temp
  
  cv.temp$pred.cv <- cv + cv.temp$hr_trend
  cv.temp$resid.cv <- cv.temp$mean  - cv.temp@data$pred.cv
  
  save(cv.temp, file=paste('data/cv_hr_fold_', val_fold, '.rda', sep=""))
  load(paste('data/cv_hr_fold_', val_fold, '.rda', sep=""))
  
  rmse_total = sqrt(sum((cv.temp$resid.cv)^2, na.rm = T)/(length(cv.temp$resid.cv[!is.na(cv.temp$resid.cv)])))
  print(rmse_total)
  # 1.05898 # 1.128003 # 1.269961 # 1.199179 # 1.14743 #
  
  data = cv.temp@data[complete.cases(cv.temp@data[, c(4, 7)]),]
  tss = t(data$mean - mean(data$mean)) %*% (data$mean - mean(data$mean))
  ess = t(data$resid.cv) %*% (data$resid.cv)
  r2 = (tss-ess)/tss
  print(r2)
  # 0.9817575 # 0.9793556 # 0.9744484 # 0.9763028 # 0.9780342 #
  
  stNum = length(cv.temp@sp)
  results = matrix(nrow = stNum, ncol = 5)
  
  for (st in 1:stNum){
    rmse = sqrt(sum((cv.temp[st, ]$resid.cv)^2, na.rm = T)/(length(cv.temp[st, ]$resid.cv[!is.na(cv.temp[st, ]$resid.cv)])))
    max = max(cv.temp[st, ]$resid.cv, na.rm = T)
    min = min(cv.temp[st, ]$resid.cv, na.rm = T)
    h = cv.temp[st, , drop=F]@sp$h
    dem = cv.temp[st, , drop=F]@sp$dem
    results[st,] = c(rmse, max, min, h, dem)
  }
  colnames(results) = c("rmse", "max", "min", "h", "dem")
  
  # save(results, file=paste('data/cv_hr_per_station_', val_fold, '.rda', sep = ""))
  load(paste('data/cv_hr_per_station_', val_fold, '.rda', sep = ""))
  
  results_order = results[order(results[,1]),]
  
  head(results_order, 20)
  tail(results_order, 20)
  
  cv.temp@sp$rmse = results[,1]
  cv.temp@sp$max = results[,2]
  cv.temp@sp$min = results[,3]
  cv.temp@sp$h = results[,4]
  cv.temp@sp$dem = results[,5]
  
}

# per month #
for (m in 1:12){
  all_res <- c()
  for (i in 1:5) {
    val_fold = i
    load(paste('data/cv_hr_fold_', val_fold, '.rda', sep=""))
    all_res <- c(all_res, cv.temp[, month(cv.temp@time)==m ]$resid.cv)
  }
  rmse = sqrt(sum((all_res)^2, na.rm = T)/(length(all_res[!is.na(all_res)])))
  print(round(rmse,2))
}

### All folds plot (STRK_gloabal and STRK_Croatia) ###
for (i in 1:5) {
  val_fold = i
  load(paste('data/cv_global_fold_', val_fold, '.rda', sep=""))
  load(paste('data/cv_global_per_station_', val_fold, '.rda', sep = ""))
  cv.temp@sp$rmse = results[,1]
  cv.temp@sp$max = results[,2]
  cv.temp@sp$min = results[,3]
  cv.temp@sp$h = results[,4]
  cv.temp@sp$dem = results[,5]
  fold_plot = cv.temp@sp[!is.na(cv.temp@sp$rmse),]
  fold_plot$rmse = round(fold_plot$rmse, 8)
  fold_plot = fold_plot[order(fold_plot@data$rmse, decreasing = T), ]
  bubbles = bubbleSP(fold_plot, zcol="rmse", scale_e = 100)
  assign(paste("cv.fold_gl", i, sep=""), cv.temp)
  assign(paste("fold_plot_gl", i, sep=""), fold_plot)
  assign(paste("bubbles_gl", i, sep=""), bubbles)
}

setwd("html/")
a = plotGoogleMaps(bubbles1_gl, filename = paste('cv_global_fold_all.html', sep = ""), zcol = "rmse", colPalette="red", add=T)
b = plotGoogleMaps(bubbles2_gl, filename = paste('cv_global_fold_all.html', sep = ""), zcol = "rmse", colPalette="green", add=T, previousMap=a)
c = plotGoogleMaps(bubbles3_gl, filename = paste('cv_global_fold_all.html', sep = ""), zcol = "rmse", colPalette="blue", add=T, previousMap=b)
d = plotGoogleMaps(bubbles4_gl, filename = paste('cv_global_fold_all.html', sep = ""), zcol = "rmse", colPalette="cyan", add=T, previousMap=c)
e = plotGoogleMaps(bubbles5_gl, filename = paste('cv_global_fold_all.html', sep = ""), zcol = "rmse", colPalette="yellow", previousMap=d)
setwd(wd)

gl_p_fold <- ggplot() +
  geom_polygon(data = border, aes(x = long, y = lat, group = group), alpha = 0.8, color = "black", fill="white", size = 0.1) +
  geom_point(data = as.data.frame(cv.fold_gl1@sp), aes(x = lon, y = lat, color = "red", size = rmse), alpha=0.6) +
  geom_point(data = as.data.frame(cv.fold_gl2@sp), aes(x = lon, y = lat, color = "green", size = rmse), alpha=0.6) +
  geom_point(data = as.data.frame(cv.fold_gl3@sp), aes(x = lon, y = lat, color = "blue", size = rmse), alpha=0.6) +
  geom_point(data = as.data.frame(cv.fold_gl4@sp), aes(x = lon, y = lat, color = "cyan", size = rmse), alpha=0.6) +
  geom_point(data = as.data.frame(cv.fold_gl5@sp), aes(x = lon, y = lat, color = "yellow", size = rmse), alpha=0.6) +
  theme(plot.title = element_text(hjust = 0.5),
        axis.text = element_text(size = 5),
        axis.title = element_text(size = 7),
        text = element_text(size = 7),
        legend.position = "bottom",
        legend.direction = "horizontal",
        legend.key.size= unit(0.2, "cm"),
        legend.margin = unit(0, "cm"),
        legend.title = element_text(size=7, face="bold")) +
  labs(x = "Longitude", y = "Latitude") + labs(size = "RMSE") + labs(color = "Folds") +
  scale_colour_manual(name = "Folds",
                      labels = c("Fold1", "Fold2", "Fold3", "Fold4", "Fold5"),
                      values = c("red"="red", "green"="green", "blue"="blue", "cyan"="cyan", "yellow"="yellow")) +
  scale_size_identity(guide="legend", breaks = c(1, 2, 3, 4, 5, 6, 7, 8, 9))
gl_p_fold

for (i in 1:5) {
  val_fold = i
  load(paste('data/cv_hr_fold_', val_fold, '.rda', sep=""))
  load(paste('data/cv_hr_per_station_', val_fold, '.rda', sep = ""))
  cv.temp@sp$rmse = results[,1]
  cv.temp@sp$max = results[,2]
  cv.temp@sp$min = results[,3]
  cv.temp@sp$h = results[,4]
  cv.temp@sp$dem = results[,5]
  fold_plot = cv.temp@sp[!is.na(cv.temp@sp$rmse),]
  fold_plot$rmse = round(fold_plot$rmse, 8)
  bubbles = bubbleSP(fold_plot, zcol="rmse", scale_e = 100)
  assign(paste("cv.fold_hr", i, sep=""), cv.temp)
  assign(paste("fold_plot_hr", i, sep=""), fold_plot)
  assign(paste("bubbles_hr", i, sep=""), bubbles)
}

setwd("html/")
a = plotGoogleMaps(bubbles1_hr, filename = paste('cv_hr_fold_all.html', sep = ""), zcol = "rmse", colPalette="red", add=T)
b = plotGoogleMaps(bubbles2_hr, filename = paste('cv_hr_fold_all.html', sep = ""), zcol = "rmse", colPalette="green", add=T, previousMap=a)
c = plotGoogleMaps(bubbles3_hr, filename = paste('cv_hr_fold_all.html', sep = ""), zcol = "rmse", colPalette="blue", add=T, previousMap=b)
d = plotGoogleMaps(bubbles4_hr, filename = paste('cv_hr_fold_all.html', sep = ""), zcol = "rmse", colPalette="cyan", add=T, previousMap=c)
e = plotGoogleMaps(bubbles5_hr, filename = paste('cv_hr_fold_all.html', sep = ""), zcol = "rmse", colPalette="yellow", previousMap=d)
setwd(wd)

loc_p_fold <- ggplot() +
  geom_polygon(data = border, aes(x = long, y = lat, group = group), alpha = 0.8, color = "black", fill="white", size = 0.1) +
  geom_point(data = as.data.frame(cv.fold_hr1@sp), aes(x = lon, y = lat, color = "red", size = rmse), alpha=0.6) +
  geom_point(data = as.data.frame(cv.fold_hr2@sp), aes(x = lon, y = lat, color = "green", size = rmse), alpha=0.6) +
  geom_point(data = as.data.frame(cv.fold_hr3@sp), aes(x = lon, y = lat, color = "blue", size = rmse), alpha=0.6) +
  geom_point(data = as.data.frame(cv.fold_hr4@sp), aes(x = lon, y = lat, color = "cyan", size = rmse), alpha=0.6) +
  geom_point(data = as.data.frame(cv.fold_hr5@sp), aes(x = lon, y = lat, color = "yellow", size = rmse), alpha=0.6) +
  theme(plot.title = element_text(hjust = 0.5),
        axis.text = element_text(size = 5),
        axis.title = element_text(size = 7),
        text = element_text(size = 7),
        legend.position = "bottom",
        legend.direction = "horizontal",
        legend.key.size= unit(0.2, "cm"),
        legend.margin = unit(0, "cm"),
        legend.title = element_text(size=7, face="bold")) +
  labs(x = "Longitude", y = "Latitude") + labs(size = "RMSE") + labs(color = "Folds") +
  scale_colour_manual(name = "Folds",
                      labels = c("Fold1", "Fold2", "Fold3", "Fold4", "Fold5"),
                      values = c("red"="red", "green"="green", "blue"="blue", "cyan"="cyan", "yellow"="yellow")) +
  scale_size_identity(guide = "legend", breaks = c(1, 2))
loc_p_fold

tiff("plot/Fig9.tiff", width = 174, height = 105, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig9.jpeg", width = 174, height = 105, units = 'mm', res = 1200)
ggarrange(gl_p_fold, loc_p_fold, ncol=2, nrow=1, common.legend = TRUE, legend="bottom")
dev.off()

################################################################################################

################################################################################################
########### Gridded data set from STRK_global ##################################################
################################################################################################

load(file=paste('data/hr_stfdf.rda', sep=""))

var="mean_global"
sp.nmax = 30

data(tregcoef)
data(tvgms)
coef = as.vector(tregcoef$tmeanGSODECAD_noMODIS)
vario= tvgms$tmeanGSODECAD_noMODIS

i_1=c(rep(1,1),1:(daysNum -1)) # 2 dana
ip1=c(1:daysNum) ### 7 days ###

dir.create(paste(wd,"/",year,"/",var, sep=""))
dir.create(paste(wd,"/",year,"/",var, "/bbox", sep=""))
dir.create(paste(wd,"/",year,"/",var, "/crop", sep=""))

for (i in 1:daysNum){
  dir.create(paste(wd,"/",year,"/",var,"/",i, sep=""))
}

dlf = list.files(path="dem_twi/HR_tiles/dem", pattern = ".sdat")
tiles = sub(".*dem", "", sub(".sdat.*", "", dlf))
rm(dlf)

d = list.files(path=paste(year,"/",var,"/",1, sep=""), pattern = ".tif")
tiles1 = sub(".*tile", "", sub(".tif.*", "", d))
rm(d)
dif=setdiff(tiles,tiles1)
tiles=dif
rm(tiles1,dif)

hr_STFDF@sp@proj4string = CRS("+proj=longlat +datum=WGS84")

registerDoParallel(cores=detectCores()-1)
foreach(i = 1:length(tiles), .packages = c("raster","spacetime","gstat","rgdal","meteo")) %dopar% {
  
  dem <- readGDAL(paste('dem_twi/HR_tiles/dem/dem', tiles[i], ".sdat", sep = ""))
  r = raster(dem)
  names(dem) = 'dem'
  dem$twi<- readGDAL(paste('dem_twi/HR_tiles/twi/twi', tiles[i], ".sdat", sep = ""))$band1
  gg <- as(dem, "SpatialPixelsDataFrame")
  gg@proj4string = CRS("+proj=longlat +datum=WGS84")
  rm(dem) 
  
  gtt_temp <- tgeom2STFDF(gg, time = time, variable = "mean")
  
  tlm  = coef[1] + coef[2]*as.numeric(gtt_temp@data$temp_geo)+coef[3]*rep(as.numeric(gg$dem),daysNum) + coef[4]*rep(as.numeric(gg$twi),daysNum) 
  tlm=matrix(tlm,ncol=daysNum)
  rm(gtt_temp)
  
  xxx<- lapply(1:daysNum, function(i) {
    obs=as(hr_STFDF[,i_1[i]:ip1[i],'gl_res', drop=F],"STSDF")
    
    krigeST(as.formula("gl_res~1"),
            data=obs,
            newdata=STF(as(gg,"SpatialPoints"),
                        hr_STFDF@time[i],
                        hr_STFDF@endTime[i]),  
            modelList=vario,
            computeVar=FALSE)$var1.pred
  } )
  res=do.call(cbind,xxx)
  rm(xxx)
  
  temp= tlm + res
  row.names(temp)<-1:nrow(gg)
  rm(tlm, res)
  
  temp=round(temp*10)
  
  for(j in 1:daysNum){
    pre = temp[, j]
    gg@data$pred=pre
    p <- try( raster( as(gg["pred"], "SpatialPixelsDataFrame")) )
    if(inherits(p, "try-error")) {
      p <- rasterize( gg, r, "pred")
    }
    writeRaster(p, paste(wd,"/",year,"/",var,"/", j, "/","tile", tiles[i], ".tif", sep = ""), "GTiff",NAflag= -32767, datatype='INT2S', overwrite=T)
    rm(pre, p)
  }
  
}
stopImplicitCluster()

### mosiac ###

registerDoParallel(cores=detectCores()-1)
foreach(dd = 1:daysNum, .packages = c("raster","rgdal")) %dopar% {
  
  cat(" mosaic num: ",dd, "\n")
  TILES.sgrd.list <- dir(paste(wd,year,var,dd, sep="/"), pattern=".tif", full.names = F )
  setwd(paste(wd, year, var, dd,sep="/"))
  
  a = sub(".*tile_", "", sub(".tif.*", "", TILES.sgrd.list))
  b = sub("_.*", "", a)
  b = as.numeric(b)
  c = sub(".*_", "", a)
  c = as.numeric(c)
  a = cbind(TILES.sgrd.list,b,c)
  d = a[order(b,c),]
  TILES.sgrd.list = d[,1]
  ff =paste("\"",TILES.sgrd.list[1:length(TILES.sgrd.list)], "\"" ,sep="", collapse=" ") 
  
  system(paste("gdal_merge.py", " -ot Int16 -o ", paste(paste(wd,year, "/",var, "/", days[dd], sep=""), '.tif', sep = "" ) ," -n -32767 -a_nodata -32767 -co NUM_THREADS=ALL_CPUS -co TILED=YES -co BLOCKXSIZE=512 -co BLOCKYSIZE=512 -co COMPRESS=DEFLATE -co PREDICTOR=2 ", paste(wd, year, "/", var, "/", dd, "/", "*.tif", sep = ""), sep=""))
  
  ### crop by BBOX ###
  r = crop(raster(paste(wd,"/",year,"/",var,"/", days[dd], ".tif", sep = "")), border)
  writeRaster(r, paste(wd,"/",year,"/",var,"/bbox/", days[dd], ".tif", sep = ""), "GTiff",NAflag= -32767, datatype='INT2S', overwrite=T)
  
  r1 = mask(r, border)
  writeRaster(r1, paste(wd,"/",year,"/",var,"/crop/", days[dd], ".tif", sep = ""), "GTiff",NAflag= -32767, datatype='INT2S', overwrite=T)
  
  unlink(paste(wd,year,var,dd, sep="/"))
  unlink(paste(paste(wd,year, "/",var, "/", days[dd], sep=""), '.tif', sep = "" ))
}
stopImplicitCluster()

################################################################################################

################################################################################################
########### Gridded data set from STRK_Croatia #################################################
################################################################################################

load(file=paste('data/hr_stfdf.rda', sep=""))

var="mean_hr"
sp.nmax = 30

data(tregcoef)
data(tvgms)
coef = as.vector(tregcoef$tmeanHR)
vario= tvgms$tmeanHR

i_1=c(rep(1,6),1:(daysNum -6))
ip1=c(1:daysNum) ### 7 days ###

dir.create(paste(wd,"/",year,"/",var, sep=""))
dir.create(paste(wd,"/",year,"/",var, "/bbox", sep=""))
dir.create(paste(wd,"/",year,"/",var, "/crop", sep=""))

for (i in 1:daysNum){
  dir.create(paste(wd,"/",year,"/",var,"/",i, sep=""))
}

dlf = list.files(path="dem_twi/HR_tiles/dem", pattern = ".sdat")
tiles = sub(".*dem", "", sub(".sdat.*", "", dlf))
rm(dlf)

d = list.files(path=paste(year,"/",var,"/",1, sep=""), pattern = ".tif")
tiles1 = sub(".*tile", "", sub(".tif.*", "", d))
rm(d)
dif=setdiff(tiles,tiles1)
tiles=dif
rm(tiles1,dif)

hr_STFDF@sp@proj4string = CRS("+proj=longlat +datum=WGS84")

registerDoParallel(cores=detectCores()-1)
foreach(i = 1:length(tiles), .packages = c("raster","spacetime","gstat","rgdal","meteo")) %dopar% {
  
  dem <- readGDAL(paste('dem_twi/HR_tiles/dem/dem', tiles[i], ".sdat", sep = ""))
  r = raster(dem)
  names(dem) = 'dem'
  dem$twi<- readGDAL(paste('dem_twi/HR_tiles/twi/twi', tiles[i], ".sdat", sep = ""))$band1
  gg <- as(dem, "SpatialPixelsDataFrame")
  gg@proj4string = CRS("+proj=longlat +datum=WGS84")
  rm(dem) 
  
  gtt_temp <- tgeom2STFDF(gg, time = time, variable = "mean")
  
  tlm  = coef[1] + coef[2]*as.numeric(gtt_temp@data$temp_geo)+coef[3]*rep(as.numeric(gg$dem),daysNum) + coef[4]*rep(as.numeric(gg$twi),daysNum) 
  tlm=matrix(tlm,ncol=daysNum)
  rm(gtt_temp)
  
  xxx<- lapply(1:daysNum, function(i) {
    obs=as(hr_STFDF[,i_1[i]:ip1[i],'hr_res', drop=F],"STSDF")
    
    krigeST(as.formula("hr_res~1"),
            data=obs,
            newdata=STF(as(gg,"SpatialPoints"),
                        hr_STFDF@time[i],
                        hr_STFDF@endTime[i]),  
            modelList=vario,
            computeVar=FALSE)$var1.pred
  } )
  res=do.call(cbind,xxx)
  rm(xxx)
  
  temp= tlm + res
  row.names(temp)<-1:nrow(gg)
  rm(tlm, res)
  
  temp=round(temp*10)
  
  for(j in 1:daysNum){
    pre = temp[, j]
    gg@data$pred=pre
    p <- try( raster( as(gg["pred"], "SpatialPixelsDataFrame")) )
    if(inherits(p, "try-error")) {
      p <- rasterize( gg, r, "pred")
    }
    writeRaster(p, paste(wd,"/",year,"/",var,"/", j, "/","tile", tiles[i], ".tif", sep = ""), "GTiff",NAflag= -32767, datatype='INT2S', overwrite=T)
    rm(pre, p)
  }
  
}
stopImplicitCluster()

### mosiac ###
registerDoParallel(cores=detectCores()-1)
foreach(dd = 1:daysNum, .packages = c("raster","rgdal")) %dopar% {
  
  cat(" mosaic num: ",dd, "\n")
  TILES.sgrd.list <- dir(paste(wd,year,var,dd, sep="/"), pattern=".tif", full.names = F )
  setwd(paste(wd, year, var, dd,sep="/"))
  
  a = sub(".*tile_", "", sub(".tif.*", "", TILES.sgrd.list))
  b = sub("_.*", "", a)
  b = as.numeric(b)
  c = sub(".*_", "", a)
  c = as.numeric(c)
  a = cbind(TILES.sgrd.list,b,c)
  d = a[order(b,c),]
  TILES.sgrd.list = d[,1]
  ff =paste("\"",TILES.sgrd.list[1:length(TILES.sgrd.list)], "\"" ,sep="", collapse=" ") 
  
  system(paste("gdal_merge.py", " -ot Int16 -o ", paste(paste(wd,year, "/",var, "/", days[dd], sep=""), '.tif', sep = "" ) ," -n -32767 -a_nodata -32767 -co NUM_THREADS=ALL_CPUS -co TILED=YES -co BLOCKXSIZE=512 -co BLOCKYSIZE=512 -co COMPRESS=DEFLATE -co PREDICTOR=2 ", paste(wd, year, "/", var, "/", dd, "/", "*.tif", sep = ""), sep=""))
  
  ### crop by BBOX ###
  r = crop(raster(paste(wd,"/",year,"/",var,"/", days[dd], ".tif", sep = "")), border)
  writeRaster(r, paste(wd,"/",year,"/",var,"/bbox/", days[dd], ".tif", sep = ""), "GTiff",NAflag= -32767, datatype='INT2S', overwrite=T)
  
  r1 = mask(r, border)
  writeRaster(r1, paste(wd,"/",year,"/",var,"/crop/", days[dd], ".tif", sep = ""), "GTiff",NAflag= -32767, datatype='INT2S', overwrite=T)
  
  unlink(paste(wd,year,var,dd, sep="/"))
  unlink(paste(paste(wd,year, "/",var, "/", days[dd], sep=""), '.tif', sep = "" ))
}
stopImplicitCluster()

### plot 4 dates ###

r1 = raster('2008/mean_hr/crop/20080101.tif')/10
r2 = raster('2008/mean_hr/crop/20080102.tif')/10
r3 = raster('2008/mean_hr/crop/20080103.tif')/10
r4 = raster('2008/mean_hr/crop/20080104.tif')/10
r11 = raster('2008/mean_global/crop/20080101.tif')/10
r22 = raster('2008/mean_global/crop/20080102.tif')/10
r33 = raster('2008/mean_global/crop/20080103.tif')/10
r44 = raster('2008/mean_global/crop/20080104.tif')/10
rs = stack(r11,r1,r22,r2,r33,r3,r44,r4)

tiff("plot/Fi11.tiff", width = 129, height = 234, units = 'mm', res = 1200, compression = "lzw")
# jpeg("plot/Fig11.jpeg", width = 129, height = 234, units = 'mm', res = 1200)
spplot(rs, col.regions = colorRampPalette(c("blue", "deepskyblue3","white", "orange", "red"))(60), cuts=59, col = NA, sp.layout = list(border, first=FALSE),
       names.attr=c('2008-01-01 gl', '2008-01-01 cro','2008-01-02 gl', '2008-01-02 cro', '2008-01-03 gl', '2008-01-03 cro', '2008-01-04 gl', '2008-01-04 cro'),
       colorkey=list(at=seq(-15, 15, 0.5))) #RdYlBu
grid.text(expression(paste("[",degree,"C]")), x=unit(0.885, "npc"), y=unit(0.985, "npc"), rot=0)
dev.off()


################################################################################################